# List Handling
use strict;
use warnings;
our (%gui, %sensgrp, %osversion, %EAudDrv, %EAudCtr, %hostspec, %ENetAdap, %ENetAttach, %bool, %chooser);

sub fill_list_guest() {
    &addrow_log("Retrieving the list of guests from $endpoint...");
    &busy_window($gui{windowMain}, 0, 'watch');
    &clr_list_guest();
    my %guestlist;
    my @IMachine = IVirtualBox_getMachines($gui{websn});
    $guestlist{$_} = IMachine_getName($_) foreach (@IMachine);

    foreach my $machine (sort { lc($guestlist{$a}) cmp lc($guestlist{$b}) } (keys %guestlist)) {
        my $prettygname = $guestlist{$machine};
        my $status = IMachine_getAccessible($machine);
        my $ISnapshot = IMachine_getCurrentSnapshot($machine);
        $prettygname .=  ' (' . ISnapshot_getName($ISnapshot) . ')' if ($ISnapshot);

        if ($status eq 'false') {
            my $IVirtualBoxErrorInfo = IMachine_getAccessError($machine);
            $status = IVirtualBoxErrorInfo_getText($IVirtualBoxErrorInfo);
        }
        else { $status = IMachine_getState($machine); }

        my $osid = IMachine_getOSTypeId($machine);
        $prettygname .= "\n   $status";
        my $iter = $gui{liststoreGuest}->append;
        $gui{liststoreGuest}->set($iter,
                                  0, $guestlist{$machine},
                                  1, $osversion{$osid}{description},
                                  2, $machine,
                                  3, $status,
                                  4, $osid,
                                  5, IMachine_getId($machine),
                                  6, $osversion{$osid}{icon},
                                  7, $prettygname);
    }

    &addrow_log("List of guests from $endpoint retrieved.");
    &busy_window($gui{windowMain}, 1);
}

sub fill_list_vminfo() {
    my %weight = (bold => 800,
                  norm => 400);
    my $str;
    $gui{liststoreVMInfo}->clear();
    my $gref = &getsel_list_guest();
    &addrow_log("Retrieving more details of $$gref{Name}...");
    my $IVRDEServer = IMachine_getVRDEServer($$gref{IMachine});
    my @IStorageController = IMachine_getStorageControllers($$gref{IMachine});
    my $IAudioAdapter = IMachine_getAudioAdapter($$gref{IMachine});
    my $IUSBController = IMachine_getUSBController($$gref{IMachine});
    &addrow_vminfo(0, 'GENERAL', 2, $weight{bold});
    &addrow_vminfo(0, 'Name:', 1, $$gref{Name}, 2, $weight{norm});
    &addrow_vminfo(0, 'OS Type:', 1, $$gref{Os}, 2, $weight{norm});
    &addrow_vminfo(0, 'SYSTEM', 2, $weight{bold});
    &addrow_vminfo(0, 'Firmware:', 1, IMachine_getFirmwareType($$gref{IMachine}), 2, $weight{norm});
    &addrow_vminfo(0, 'Base Memory:', 1, IMachine_getMemorySize($$gref{IMachine}) . ' MiB', 2, $weight{norm});
    &addrow_vminfo(0, 'Processor(s):', 1, IMachine_getCPUCount($$gref{IMachine}), 2, $weight{norm});
    $str = '';

    foreach (1..4) {
        my $bdev = IMachine_getBootOrder($$gref{IMachine}, $_);
        $str .= "$bdev, " if ($bdev ne 'Null');
    }

    $str =~ s/, $//;
    &addrow_vminfo(0, 'Boot Order:', 1, $str, 2, $weight{norm}) if ($str);
    $str = '';
    $str .= 'VT-x/AMD-V, ' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'Enabled') eq 'true');
    $str .= 'VPID, ' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'VPID') eq 'true');
    $str .= 'PAE/NX, ' if (IMachine_getCPUProperty($$gref{IMachine}, 'PAE') eq 'true');
    $str .= 'Nested Paging' if (IMachine_getHWVirtExProperty($$gref{IMachine}, 'NestedPaging') eq 'true');
    $str =~ s/, $//;
    &addrow_vminfo(0, 'Acceleration:', 1, $str, 2, $weight{norm}) if ($str);
    &addrow_vminfo(0, 'DISPLAY', 2, $weight{bold});
    &addrow_vminfo(0, 'Video Memory:', 1, IMachine_getVRAMSize($$gref{IMachine}) . ' MiB', 2, $weight{norm});
    &addrow_vminfo(0, 'Screens: ', 1, IMachine_getMonitorCount($$gref{IMachine}), 2, $weight{norm});
    $str = '';
    $str .= '2D Video, ' if (IMachine_getAccelerate2DVideoEnabled($$gref{IMachine}) eq 'true');
    $str .= '3D' if (IMachine_getAccelerate3DEnabled($$gref{IMachine}) eq 'true');
    &addrow_vminfo(0, 'Display Acceleration:', 1, $str, 2, $weight{norm}) if ($str);
    &addrow_vminfo(0, 'Remote Display Port(s):', 1, IVRDEServer_getVRDEProperty($IVRDEServer, 'TCP/Ports'), 2, $weight{norm}) if (IVRDEServer_getEnabled($IVRDEServer) eq 'true');
    &addrow_vminfo(0, 'STORAGE', 2, $weight{bold});

    foreach my $controller (@IStorageController) {
        my $controllername = IStorageController_getName($controller);
        &addrow_vminfo(0, 'Controller:', 1, $controllername, 2, $weight{norm});
        my @IMediumAttachment = IMachine_getMediumAttachmentsOfController($$gref{IMachine}, $controllername);
        foreach my $attachment (@IMediumAttachment) {
            if ($$attachment{medium}) {
                IMedium_refreshState($$attachment{medium}); # Needed to bring in current sizes
                # Use the base medium for information purposes
                my $size = &bytesToX(IMedium_getLogicalSize($$attachment{medium}));
                &addrow_vminfo(0, "  Port $$attachment{port}:",
                               1, IMedium_getName(IMedium_getBase($$attachment{medium})) . " ($$attachment{type}, $size)",
                               2, $weight{norm});
            }
        }
    }

    &addrow_vminfo(0, 'AUDIO', 2, $weight{bold});

    if (IAudioAdapter_getEnabled($IAudioAdapter) eq 'true') {
        &addrow_vminfo(0, 'Host Driver:', 1, $EAudDrv{IAudioAdapter_getAudioDriver($IAudioAdapter)}{desc}, 2, $weight{norm});
        &addrow_vminfo(0, 'Controller:', 1, $EAudCtr{IAudioAdapter_getAudioController($IAudioAdapter)}, 2, $weight{norm});
    }
    else { &addrow_vminfo(0, 'Audio Disabled', 2, $weight{norm}); }

    &addrow_vminfo(0, 'NETWORK', 2, $weight{bold});

    foreach (0..($hostspec{maxnet}-1)) {
        my $INetworkAdapter = IMachine_getNetworkAdapter($$gref{IMachine}, $_);
        if (INetworkAdapter_getEnabled($INetworkAdapter) eq 'true') {
            my $attachtype = INetworkAdapter_getAttachmentType($INetworkAdapter);
            $str = $ENetAdap{INetworkAdapter_getAdapterType($INetworkAdapter)} . ' (' . $ENetAttach{$attachtype};

            if ($attachtype eq 'Bridged' or $attachtype eq 'HostOnly') { $str .= ', ' . INetworkAdapter_getHostInterface($INetworkAdapter); }
            elsif ($attachtype eq 'Internal') { $str .= ', ' . INetworkAdapter_getInternalNetwork($INetworkAdapter); }

            $str .= ')';

            &addrow_vminfo(0, "Adapter $_:", 1, $str, 2, $weight{norm});
        }
    }

    &addrow_vminfo(0, 'SERIAL PORTS', 2, $weight{bold});

    foreach (0..($hostspec{maxser}-1)) {
        my $ISerialPort = IMachine_getSerialPort($$gref{IMachine}, $_);
        if (ISerialPort_getEnabled($ISerialPort) eq 'true') {
            &addrow_vminfo(0, "Port #$_:",
                           1, 'Enabled, ' . ISerialPort_getHostMode($ISerialPort) . ', ' . ISerialPort_getPath($ISerialPort),
                           2, $weight{norm});
        }
        else { &addrow_vminfo(0, "Port #$_:", 1, 'Disabled', 2, $weight{norm}); }
    }

    &addrow_vminfo(0, 'PARALLEL PORT', 2, $weight{bold});
    my $IParallelPort = IMachine_getParallelPort($$gref{IMachine}, 0);

    if (IParallelPort_getEnabled($IParallelPort) eq 'true') {
        &addrow_vminfo(0, 'LPT:',
                       1, 'Enabled, ' . IParallelPort_getPath($IParallelPort),
                       2, $weight{norm});
    }
    else { &addrow_vminfo(0, 'Disabled', 2, $weight{norm}); }

    &addrow_vminfo(0, 'USB', 2, $weight{bold});

    if (IUSBController_getEnabled($IUSBController) eq 'true' ) {
        my @IUSBDeviceFilter = IUSBController_getDeviceFilters($IUSBController);
        my $active = 0;
        foreach (@IUSBDeviceFilter) {
            $active++ if (IUSBDeviceFilter_getActive($_) eq 'true');
        }
        &addrow_vminfo(0, 'Device Filters:', 1,  scalar(@IUSBDeviceFilter) . " ($active active)", 2, $weight{norm});
    }
    else { &addrow_vminfo(0, 'Disabled', 2, $weight{norm}); }

    my @sf = IMachine_getSharedFolders($$gref{IMachine});
    &addrow_vminfo(0, 'SHARED FOLDERS', 2, $weight{bold});
    &addrow_vminfo(0, 'Shared Folders:', 1, scalar(@sf), 2, $weight{norm});
    &addrow_log("More Details of $$gref{Name} retrieved.");
}

sub fill_list_snapshots() {
    &busy_window($gui{windowMain}, 0, 'watch');
    &clr_list_snapshots();
    my $gref = &getsel_list_guest();
    my $ISnapshot_current = IMachine_getCurrentSnapshot($$gref{IMachine});
    &addrow_log("Retrieving snapshot information for $$gref{Name}...");

    if (IMachine_getSnapshotCount($$gref{IMachine}) > 0) {
        my $ISnapshot = IMachine_findSnapshot($$gref{IMachine}, undef); # get first snapshot
        &recurse_snapshot($ISnapshot, undef, $ISnapshot_current);
        $gui{treeviewSnapshots}->expand_all();
    }

    &addrow_log("Snapshot information for $$gref{Name} retrieved.");
    &busy_window($gui{windowMain}, 1);
}

sub fill_list_remotefiles_ext() {
    my ($location) = @_;

    $gui{liststoreRemoteFileChooser}->clear();
    IVFSExplorer_cd($gui{IVFSExplorer}, $location);
    IVFSExplorer_update($gui{IVFSExplorer});
    my @entries = IVFSExplorer_entryList($gui{IVFSExplorer});
    my $chop = (@entries / 4);
    my %files;
    unshift @entries, 'Dummy';

    foreach my $x (1..$chop) {
        $files{$entries[$x]}{type} = $entries[$x + $chop];
        $files{$entries[$x]}{size} = &bytesToX($entries[$x + ($chop * 2)]);
        $files{$entries[$x]}{mode} = sprintf("%o", $entries[$x + ($chop * 3)]);
    }

    my $iter = $gui{liststoreRemoteFileChooser}->append();
    $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Parent)', 1, '..', 2, '', 3, '');

    foreach my $fname (sort { lc($a) cmp lc($b) } (keys %files)) {
        # Only interested in directories and bmp files
        if ($files{$fname}{type} == 4) {
            my $iter = $gui{liststoreRemoteFileChooser}->append();
            $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Dir)', 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode});
        }
        elsif ($fname =~ m/\.$chooser{ext}$/i) {
            my $iter = $gui{liststoreRemoteFileChooser}->append();
            $gui{liststoreRemoteFileChooser}->set($iter, 0, "($chooser{ext})", 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode});
        }
    }

    $gui{entryRemoteFileChooserLocation}->set_text(IVFSExplorer_getPath($gui{IVFSExplorer}));
}

sub fill_list_remotefolders() {
    my ($location) = @_;

    $gui{liststoreRemoteFileChooser}->clear();
    IVFSExplorer_cd($gui{IVFSExplorer}, $location);
    IVFSExplorer_update($gui{IVFSExplorer});
    my @entries = IVFSExplorer_entryList($gui{IVFSExplorer});
    my $chop = (@entries / 4);
    my %files;
    unshift @entries, 'Dummy';

    foreach my $x (1..$chop) {
        $files{$entries[$x]}{type} = $entries[$x + $chop];
        $files{$entries[$x]}{size} = &bytesToX($entries[$x + ($chop * 2)]);
        $files{$entries[$x]}{mode} = sprintf("%o", $entries[$x + ($chop * 3)]);
    }

    my $iter = $gui{liststoreRemoteFileChooser}->append();
    $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Parent)', 1, '..', 2, '', 3, '');

    foreach my $fname (sort { lc($a) cmp lc($b) } (keys %files)) {
        next if ($files{$fname}{type} != 4); # Only interested in directories
        my $iter = $gui{liststoreRemoteFileChooser}->append();
        $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Dir)', 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode});
    }

    $gui{entryRemoteFileChooserLocation}->set_text(IVFSExplorer_getPath($gui{IVFSExplorer}));
}

sub fill_list_remotefiles() {
    my ($location) = @_;
    $gui{liststoreRemoteFileChooser}->clear();
    IVFSExplorer_cd($gui{IVFSExplorer}, $location);
    IVFSExplorer_update($gui{IVFSExplorer});
    my @entries = IVFSExplorer_entryList($gui{IVFSExplorer});
    my $chop = (@entries / 4);
    my %files;
    unshift @entries, 'Dummy';

    foreach my $x (1..$chop) {
        $files{$entries[$x]}{type} = $entries[$x + $chop];
        $files{$entries[$x]}{size} = &bytesToX($entries[$x + ($chop * 2)]);
        $files{$entries[$x]}{mode} = sprintf("%o", $entries[$x + ($chop * 3)]);
    }

    my $iter = $gui{liststoreRemoteFileChooser}->append();
    $gui{liststoreRemoteFileChooser}->set($iter, 0, '(Parent)', 1, '..', 2, '', 3, '');

    foreach my $fname (sort { lc($a) cmp lc($b) } (keys %files)) {
        my $type = '';
        $type = '(Dir)' if ($files{$fname}{type} == 4);
        my $iter = $gui{liststoreRemoteFileChooser}->append();
        $gui{liststoreRemoteFileChooser}->set($iter, 0, $type, 1, $fname, 2, $files{$fname}{size}, 3, $files{$fname}{mode});
    }

    $gui{entryRemoteFileChooserLocation}->set_text(IVFSExplorer_getPath($gui{IVFSExplorer}));
}

sub fill_list_serverinfo() {
    $gui{liststoreServerInfo}->clear();
    &addrow_serverinfo(0, 'URL:', 1, $endpoint);
    &addrow_serverinfo(0, 'VirtualBox Version:', 1, $hostspec{vbver});
    &addrow_serverinfo(0, 'Build Revision:', 1, $hostspec{buildrev});
    &addrow_serverinfo(0, 'Package Type:', 1, $hostspec{pkgtype});
    &addrow_serverinfo(0, 'Global Settings File:', 1, $hostspec{settingsfile});
    &addrow_serverinfo(0, 'Machine Folder:', 1, $hostspec{machinedir});
    &addrow_serverinfo(0, 'Server Logical CPUs:', 1, $hostspec{maxhostcpuon});
    &addrow_serverinfo(0, 'Server CPU Type:', 1, $hostspec{cpudesc});
    &addrow_serverinfo(0, 'Server CPU Speed:', 1, "$hostspec{cpuspeed} Mhz (approx)");
    &addrow_serverinfo(0, 'Server Memory Size:', 1, "$hostspec{memsize} MiB");
    &addrow_serverinfo(0, 'Server OS:', 1, $hostspec{os});
    &addrow_serverinfo(0, 'Server OS Version:', 1, $hostspec{osver});
    &addrow_serverinfo(0, 'Min Guest RAM:', 1, "$hostspec{minguestram} MiB");
    &addrow_serverinfo(0, 'Max Guest RAM:', 1, "$hostspec{maxguestram} MiB");
    &addrow_serverinfo(0, 'Min Guest Video RAM:', 1, "$hostspec{minguestvram} MiB");
    &addrow_serverinfo(0, 'Max Guest Video RAM:', 1, "$hostspec{maxguestvram} MiB");
    &addrow_serverinfo(0, 'Max Guest CPUs:', 1, $hostspec{maxguestcpu});
    &addrow_serverinfo(0, 'Max HD Image Size:', 1, &bytesToX($hostspec{maxhdsize}));
}

sub fill_list_vmmhd() {
    &busy_window($gui{dialogVMM}, 0, 'watch');
    $gui{labelVMMAttachedToField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMLocationField}->set_text(''); # Do whenever list is cleared
    $gui{treestoreVMMHD}->clear();
    my $IMediumRef = &get_all_media('HardDisk');

    foreach (sort { lc($a) cmp lc($b) } (keys %$IMediumRef)) {
        &recurse_media_snapshot($gui{treestoreVMMHD}, $$IMediumRef{$_}, undef, 'HardDisk');
    }

    &busy_window($gui{dialogVMM}, 1);
}

sub fill_list_vmmdvd() {
    &busy_window($gui{dialogVMM}, 0, 'watch');
    $gui{labelVMMAttachedToField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMLocationField}->set_text(''); # Do whenever list is cleared
    $gui{treestoreVMMDVD}->clear();
    my $IMediumRef = &get_all_media('DVD');

    foreach (sort { lc($a) cmp lc($b) } (keys %$IMediumRef)) {
        &recurse_media_snapshot($gui{treestoreVMMDVD}, $$IMediumRef{$_}, undef, 'DVD');
    }

    &busy_window($gui{dialogVMM}, 1);
}

sub fill_list_vmmfloppy() {
    &busy_window($gui{dialogVMM}, 0, 'watch');
    $gui{labelVMMAttachedToField}->set_text(''); # Do whenever list is cleared
    $gui{labelVMMLocationField}->set_text(''); # Do whenever list is cleared
    $gui{treestoreVMMFloppy}->clear();
    my $IMediumRef = &get_all_media('Floppy');

    foreach (sort { lc($a) cmp lc($b) } (keys %$IMediumRef)) {
        &recurse_media_snapshot($gui{treestoreVMMFloppy}, $$IMediumRef{$_}, undef, 'Floppy');
    }

    &busy_window($gui{dialogVMM}, 1);
}

sub fill_list_editshared() {
    my ($IMachine) = @_;
    &sensitive_off($gui{buttonEditSharedRemove}, $gui{buttonEditSharedEdit}); # Do whenever list is cleared
    $gui{liststoreEditShared}->clear();
    my @ISharedFolder = IMachine_getSharedFolders($IMachine);

    foreach my $share (@ISharedFolder) {
        my $access = 'Full';
        my $automount = 'No';
        my $tooltip = "$$share{name}: $$share{hostPath}";
        $access = 'Read-Only' if ($$share{writable} eq 'false');
        $automount = 'Yes' if ($$share{autoMount} eq 'true');
        $tooltip = $$share{lastAccessError} if ($$share{lastAccessError});
        $tooltip = 'Share is not accessible' if ($$share{accessible} eq 'false');
        my $iter = $gui{liststoreEditShared}->append;
        if ($$share{accessible} eq 'false') { $gui{liststoreEditShared}->set($iter, 0, $$share{name}, 1, $$share{hostPath}, 2, $access, 3, $automount, 4, $gui{pixbufstatuserror16}, 5, $tooltip); }
        else { $gui{liststoreEditShared}->set($iter, 0, $$share{name}, 1, $$share{hostPath}, 2, $access, 3, $automount, 5, $tooltip); }
    }
}

sub fill_list_editstorage() {
    my ($IMachine) = @_;
    &busy_window($gui{dialogEdit}, 0, 'watch');
    &sensitive_off(@{ $sensgrp{stor_unsel} }); # Do whenever list is cleared
    $gui{treestoreEditStor}->clear();
    my @IStorageController = IMachine_getStorageControllers($IMachine);

    foreach my $controller (@IStorageController) {
        my $cname = IStorageController_getName($controller);
        my $bus = IStorageController_getBus($controller);
        my $iter = $gui{treestoreEditStor}->append(undef);
        $gui{treestoreEditStor}->set($iter, 0, $cname, 1, $cname, 2, $bus . ' Controller', 3, $controller, 7, $cname);
        my @IMediumAttachment = IMachine_getMediumAttachmentsOfController($IMachine, $cname);

        foreach my $attachment (@IMediumAttachment) {
            my $citer = $gui{treestoreEditStor}->append($iter);

            if ($$attachment{medium}) {
                my $baseIMedium = IMedium_getBase($$attachment{medium});
                my $mname = IMedium_getName($baseIMedium); # Use the
                my $prettyname = $mname;
                if ($baseIMedium ne $$attachment{medium}) { $prettyname = "(*) $prettyname"; }
                $gui{treestoreEditStor}->set($citer, 0, $mname,
                                                     1, $cname,
                                                     2, $$attachment{type},
                                                     3, $$attachment{medium},
                                                     4, $$attachment{device},
                                                     5, $$attachment{port},
                                                     6, IMedium_getLocation($$attachment{medium}),
                                                     7, $prettyname);
            }
            else {
                $gui{treestoreEditStor}->set($citer, 0, '<empty>',
                                                     1, $cname,
                                                     2, $$attachment{type},
                                                     3, '<empty>',
                                                     4, $$attachment{device},
                                                     5, $$attachment{port},
                                                     6, '<empty>',
                                                     7, '<empty>');
            }
        }
    }

    $gui{treeviewEditStor}->expand_all();
    &busy_window($gui{dialogEdit}, 1);
}

sub fill_list_usbfilters() {
    my ($IUSBController) = @_;
    &busy_window($gui{dialogVMM}, 0, 'watch');
    &sensitive_off(@{ $sensgrp{usb_sel} }); # Do whenever list is cleared
    $gui{liststoreEditUSBFilter}->clear();
    my @IUSBDeviceFilters = IUSBController_getDeviceFilters($IUSBController);
    my $pos = 0;

    foreach my $filter (@IUSBDeviceFilters) {
        my $iter = $gui{liststoreEditUSBFilter}->append();
        $gui{liststoreEditUSBFilter}->set($iter,
                                      0, $bool{IUSBDeviceFilter_getActive($filter)},
                                      1, $filter,
                                      2, IUSBDeviceFilter_getName($filter),
                                      3, $pos);
        $pos++;
    }

    &busy_window($gui{dialogVMM}, 1);
}

sub fill_list_vbprefsnet() {
    &busy_window($gui{dialogVBPrefs}, 0, 'watch');
    &sensitive_off(@{ $sensgrp{vbprefsnet} }); # Do whenever list is cleared
    $gui{liststoreVBPrefsNet}->clear();
    my $IHost = IVirtualBox_getHost($gui{websn});
    my @IHostNetworkInterface = IHost_findHostNetworkInterfacesOfType($IHost, 'HostOnly');

    foreach my $if (@IHostNetworkInterface) {
        my $iter = $gui{liststoreVBPrefsNet}->append();
        $gui{liststoreVBPrefsNet}->set($iter,
                                       0, IHostNetworkInterface_getName($if),
                                       1, $if,
                                       2, IHostNetworkInterface_getId($if));
    }
    &busy_window($gui{dialogVBPrefs}, 1);
}

sub clr_list_guest() {
    &sensitive_off(@{ $sensgrp{unselected} });
    $gui{liststoreGuest}->clear();
    &clr_summarydetails();
    &clr_list_snapshots();
}

sub clr_list_snapshots() {
    &sensitive_off(@{ $sensgrp{snap} });
    $gui{treestoreSnapshots}->clear();
}

# Adds a message to message log and scrolls to bottom
sub addrow_log() {
    my ($msg) = @_;
    my $iter = $gui{liststoreLog}->append;
    $gui{liststoreLog}->set($iter, 0, $msg);
    $gui{treeviewLog}->scroll_to_cell($gui{treeviewLog}->get_model->get_path($iter));
}

sub addrow_vminfo() {
    my $iter = $gui{liststoreVMInfo}->append;
    $gui{liststoreVMInfo}->set($iter, @_);
    return $iter;
}

sub addrow_serverinfo() {
    my $iter = $gui{liststoreServerInfo}->append;
    $gui{liststoreServerInfo}->set($iter, @_);
    return $iter;
}

sub getsel_combo() {
    my ($widget, $col) = @_;
    my $model = $widget->get_model();
    my $iter = $widget->get_active_iter();

    if (defined($col)) { return $model->get($iter, $col); }
    else { return $model->get($iter); }
}

sub getsel_list_guest() {
    my $model = $gui{treeviewGuest}->get_model();
    my ($path) = $gui{treeviewGuest}->get_cursor();
    my $iter = $gui{treeviewGuest}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Os', 'IMachine', 'Status', 'Osid', 'Uuid', 'Icon', 'Prettyname');
    return \%hash;
}

sub getsel_list_remotefiles() {
    my $model = $gui{treeviewRemoteFileChooser}->get_model();
    my ($path) = $gui{treeviewRemoteFileChooser}->get_cursor();
    my $iter = $gui{treeviewRemoteFileChooser}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Type', 'FileName', 'Size', 'Mode');
    return \%hash;
}

sub getsel_list_snapshots() {
    my $model = $gui{treeviewSnapshots}->get_model();
    my ($path) = $gui{treeviewSnapshots}->get_cursor();
    my $iter = $gui{treeviewSnapshots}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Date', 'ISnapshot');
    return \%hash;
}

sub getsel_list_vmmhd() {
    my $model = $gui{treeviewVMMHD}->get_model();
    my ($path) = $gui{treeviewVMMHD}->get_cursor();
    my $iter = $gui{treeviewVMMHD}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IMedium', 'Attached', 'Asize', 'Vsize', 'Accessible', 'Tooltip');
    return \%hash;
}

sub getsel_list_vmmdvd() {
    my $model = $gui{treeviewVMMDVD}->get_model();
    my ($path) = $gui{treeviewVMMDVD}->get_cursor();
    my $iter = $gui{treeviewVMMDVD}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IMedium', 'Attached', 'Size', 'Accessible', 'Tooltip');
    return \%hash;
}

sub getsel_list_vmmfloppy() {
    my $model = $gui{treeviewVMMFloppy}->get_model();
    my ($path) = $gui{treeviewVMMFloppy}->get_cursor();
    my $iter = $gui{treeviewVMMFloppy}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IMedium', 'Attached', 'Size', 'Accessible', 'Tooltip');
    return \%hash;
}

sub getsel_list_editshared() {
    my $model = $gui{treeviewEditShared}->get_model();
    my ($path) = $gui{treeviewEditShared}->get_cursor();
    my $iter = $gui{treeviewEditShared}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Folder', 'Access', 'Mount', 'Accessible', 'Tooltip');
    return \%hash;
}

sub getsel_list_editstorage() {
    my $model = $gui{treeviewEditStor}->get_model();
    my ($path) = $gui{treeviewEditStor}->get_cursor();
    my $iter = $gui{treeviewEditStor}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'Controllernameformedia', 'Type', 'IObject', 'Devnum', 'Port', 'Location', 'Prettyname');
    return \%hash;
}

sub getsel_list_usbfilters() {
    my $model = $gui{treeviewEditUSBFilters}->get_model();
    my ($path) = $gui{treeviewEditUSBFilters}->get_cursor();
    my $iter = $gui{treeviewEditUSBFilters}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Enabled', 'IUSBDeviceFilter', 'Name', 'Position');
    return \%hash;
}

sub getsel_list_vbprefsnet() {
    my $model = $gui{treeviewVBPrefsNet}->get_model();
    my ($path) = $gui{treeviewVBPrefsNet}->get_cursor();
    my $iter = $gui{treeviewVBPrefsNet}->get_selection->get_selected() ? $model->get_iter($path) : $model->get_iter_first();
    return undef if (!$iter);
    my @row = $model->get($iter);
    my %hash;
    $hash{$_} = shift @row foreach ('Name', 'IHostNetworkInterface', 'Uuid');
    return \%hash;
}

sub onsel_list_guest() {
    my $gref = &getsel_list_guest();
    &fill_summarydetails();
    &sensitive_off(@{ $sensgrp{unselected} });
    my $status = IMachine_getState($$gref{IMachine});
    &fill_list_snapshots();

    if ($status eq 'Running' | $status eq 'Starting') { &sensitive_on(@{ $sensgrp{running} }); }
    elsif ($status eq 'Saved') { &sensitive_on(@{ $sensgrp{saved} }); }
    elsif ($status eq 'Paused') { &sensitive_on(@{ $sensgrp{paused} }); }
    elsif ($status eq 'PoweredOff' | $status eq 'Aborted') { &sensitive_on(@{ $sensgrp{poweroff} }); }
    else { &sensitive_off(@{ $sensgrp{unselected} }); }

}

sub onsel_list_remotefiles() {
    my $fileref = &getsel_list_remotefiles();

    if ($$fileref{FileName} eq '..') { &cdup_remotefilechooser(); }
    elsif ($$fileref{Type} eq '(Dir)') {
        my $path = IVFSExplorer_getPath($gui{IVFSExplorer});
        IVFSExplorer_cd($gui{IVFSExplorer}, "$path/$$fileref{FileName}");
        &{$chooser{populator}}(IVFSExplorer_getPath($gui{IVFSExplorer}));
    }
}

sub onsel_list_snapshots() { &sensitive_on(@{ $sensgrp{snap} }); }

sub onsel_list_vmmhd() {
    my $hdref = &getsel_list_vmmhd();

    # Don't allow remove/release if it has sub-snapshots
    if (IMedium_getChildren($$hdref{IMedium})) {
        &sensitive_off($gui{toolbuttonVMMRemove});
        &sensitive_off($gui{toolbuttonVMMRelease});
    }
    elsif ($$hdref{Attached} eq '<None>') {
        &sensitive_on($gui{toolbuttonVMMRemove});
        &sensitive_off($gui{toolbuttonVMMRelease});
    }
    else {
        &sensitive_off($gui{toolbuttonVMMRemove});
        &sensitive_on($gui{toolbuttonVMMRelease});
    }

    $gui{labelVMMAttachedToField}->set_text($$hdref{Attached});
    $gui{labelVMMLocationField}->set_text(IMedium_getLocation($$hdref{IMedium}));
}

sub onsel_list_vmmdvd() {
    my $dvdref = &getsel_list_vmmdvd();

    # Don't allow remove/release if it has sub-snapshots
    if (IMedium_getChildren($$dvdref{IMedium})) {
        &sensitive_off($gui{toolbuttonVMMRemove});
        &sensitive_off($gui{toolbuttonVMMRelease});
    }
    elsif ($$dvdref{Attached} eq '<None>') {
        &sensitive_on($gui{toolbuttonVMMRemove});
        &sensitive_off($gui{toolbuttonVMMRelease});
    }
    else {
        &sensitive_off($gui{toolbuttonVMMRemove});
        &sensitive_on($gui{toolbuttonVMMRelease});
    }

    $gui{labelVMMAttachedToField}->set_text($$dvdref{Attached});
    $gui{labelVMMLocationField}->set_text(IMedium_getLocation($$dvdref{IMedium}));
}

sub onsel_list_vmmfloppy() {
    my $floppyref = &getsel_list_vmmfloppy();

    # Don't allow remove/release if it has sub-snapshots
    if (IMedium_getChildren($$floppyref{IMedium})) {
        &sensitive_off($gui{toolbuttonVMMRemove});
        &sensitive_off($gui{toolbuttonVMMRelease});
    }
    elsif ($$floppyref{Attached} eq '<None>') {
        &sensitive_on($gui{toolbuttonVMMRemove});
        &sensitive_off($gui{toolbuttonVMMRelease});
    }
    else {
        &sensitive_off($gui{toolbuttonVMMRemove});
        &sensitive_on($gui{toolbuttonVMMRelease});
    }

    $gui{labelVMMAttachedToField}->set_text($$floppyref{Attached});
    $gui{labelVMMLocationField}->set_text(IMedium_getLocation($$floppyref{IMedium}));
}

sub onsel_list_editshared() { &sensitive_on($gui{buttonEditSharedRemove}, $gui{buttonEditSharedEdit}); }

sub onsel_list_editstorage() {
    my $storref = &getsel_list_editstorage();

    if ($$storref{Type} =~ m/Controller/) {
        &sensitive_on(@{ $sensgrp{ctr_sel_on} });
        &sensitive_off($gui{buttonEditStorRemoveAttach});
        &hide_off($gui{frameEditStorAttr});
        $gui{entryEditStorCtrName}->set_text($$storref{Name});
        &storage_ctrtype_fill($$storref{Type}, $$storref{IObject});

        if ($$storref{Type} eq 'IDE Controller' or $$storref{Type} eq 'SATA Controller') {
            &sensitive_on($gui{menuitemAttachHD}, $gui{menuitemAttachDVD});
            &sensitive_off($gui{menuitemAttachFloppy});
        }
        elsif ($$storref{Type} eq 'Floppy Controller') {
            &sensitive_off($gui{menuitemAttachHD}, $gui{menuitemAttachDVD});
            &sensitive_on($gui{menuitemAttachFloppy});
        }
        else {
            &sensitive_on($gui{menuitemAttachHD});
            &sensitive_off($gui{menuitemAttachFloppy}, $gui{menuitemAttachDVD});
        }
    }
    elsif ($$storref{Type} eq 'DVD') { # Allow DVDs to be inserted if DVD drive selected
        &sensitive_on($gui{buttonEditStorRemoveAttach}, $gui{buttonEditStorAddAttach}, $gui{menuitemAttachDVD});
        &sensitive_off($gui{menuitemAttachHD}, $gui{menuitemAttachFloppy}, $gui{buttonEditStorRemoveCtr});
        &hide_on($gui{frameEditStorAttr});
    }
    elsif ($$storref{Type} eq 'Floppy') { # Allow floppys to be inserted if floppy drive selected
        &sensitive_on($gui{buttonEditStorRemoveAttach}, $gui{buttonEditStorAddAttach}, $gui{menuitemAttachFloppy});
        &sensitive_off($gui{menuitemAttachHD}, $gui{menuitemAttachDVD}, $gui{buttonEditStorRemoveCtr});
        &hide_on($gui{frameEditStorAttr});
    }
    else {
        &sensitive_on($gui{buttonEditStorRemoveAttach});
        &sensitive_off(@{ $sensgrp{ctr_sel_off} });
        &hide_on($gui{frameEditStorAttr});
    }
}

sub onsel_list_usbfilters() { &sensitive_on( @{ $sensgrp{usb_sel} } ); }

sub onsel_list_vbprefsnet() { &sensitive_on( @{ $sensgrp{vbprefsnet} } ); }

1;


# ***
# Some versions of perl-Gtk2 (for example 1.223) appear to emit the "cursor-changed"
# signal for treeviews, when in fact the cursor hasn't changed. This leads to failure
# when trying to get the selected item because there isn't one! So I default to returning
# the first entry for "phantom" cursor changes. No doubt there's a better way of doing this
